% Gets the diff between two Simulink systems (model/library)
%
% Function computes a diff between two Simulink systems as string message
% and as struct. The function is especially useful for testing
% transformations on Simulink systems. For such tests you can save two
% systems: One system which is the system before the transformation and one
% system which specifies how the system should look like after the
% transformation. Then you can do the transformation of the first system
% and compare the result to the second system using this diff function. The
% test can just assert an empty diff message string.
%
% Currently, the function computes the diff between Simulink parts
% only, which means that Stateflow is excluded. It tries to match the
% following elements between old and new system: blocks, annotations and
% lines. Blocks and annotations are regarded as the same if they have the
% same full path (the text of an annotation is part of full path). Lines
% are regarded as the same if they have the same source port and
% destination ports.
%
% Everything that is found in new system that could not be matched to
% elements from old system is regarded as new. Everything that is found in
% old system that could not be matched to elements from new system is
% regared as deleted.
%
% If elements from a subsystem could not be matched uniquely between the
% two systems (like two lines with the same source port but no destination
% port) and the number of these ambiguous elements differ between the old
% and new system, an 'ambiguous new' or 'ambiguous deleted' group is
% returned with the number of new/deleted elements and the elements which
% could be new/deleted.
%
% In addition, the function returns detected changes of parameters between
% two elements that could be matched one-to-one between old and new system.
% All elements from the old system that are not classified as deleted but
% cannot be matched one-to-one to the new system are listed as ambiguous
% elements in the parameters part, because there could be parameter changes
% in these elements which are not detected.
%
% For blocks, the port parameters are also compared, since these also
% contain user-relevant information (e.g. DataLoggingName). In order to be
% able to distinguish between changes in the port parameters and block
% parameters in the parameters.changes struct, a prefix in the form
% 'PortType:PortNumber' is used before the parameter name (e.g.
% 'Outport:1:DataLoggingName') in this struct. This prefix must not be used
% in parametersToIgnore.
%
% For TargetLink blocks, the TargetLink properties are also compared
% (tl_get). In order to be able to distinguish between normal block
% parameters and TargetLink properties in the parameters.changes struct,
% a prefix 'TL:' is used before the parameter names (e.g.
% 'TL:output.max') in this struct. This prefix can also be used in
% parametersToIgnore. The normal block parameter 'data' is automatically
% ignored for TargetLink blocks because it is only the storage parameter
% for the TargetLink properties, which are already compared individually.
% Since TargetLink properties that are not set return the value NaN, two
% NaN values are generally regarded as equal by the differ.
%
%
% INPUT PARAMETERS:
%   oldSystem - old system specified as char array or as mes.more_mes.sl.System
%   newSystem - New system specified as char array or as mes.more_mes.sl.System
%   parametersToIgnore (optional) - Cell string with parameters which
%                                   should be ignored in detecting
%                                   parameter changes (e.g. 'position'), or
%                                   function with the signature tf =
%                                   f(oldElement, newElement,
%                                   parameterName) which determines whether
%                                   a parameter should be ignored (true) or
%                                   not (false).
% OUTPUT PARAMETERS:
%   diffStruct - Diff struct with following fields:
%       new.elements: List with all elements classified as new
%       new.ambiguousGroups: Struct with all ambiguous new groups with
%                            the following fields:
%                   number: Number of new elements in group
%                   elements: List with elements from which 'number' of
%                             elements are new.
%       deleted.elements: List with all elements classified as deleted
%       deleted.ambiguousGroups: Struct with all ambiguous deleted groups
%                                with the following fields:
%                   number: Number of deleted elements in group
%                   elements: List with elements from which 'number' of
%                             elements are deleted.
%       parameters.changes: Struct array with all parameter changes with
%                          the following fields:
%                   oldElement: Element with parameter change in oldSystem
%                   newElement: Element with parameter change in newSystem
%                   parameters: Struct array with all parameters changed in
%                          element. Struct has the following fields:
%                              name: Name of parameter changed
%                              oldValue: Value of parameter in oldElement
%                              newValue: Value of parameter in newElement
%       parameter.ambiguousElementsFromOldSystem: List with all
%               elements from oldSystem which are not classified as
%               deleted but cannot be matched one-to-one to newSystem.
%               There may be parameter changes in this elements which are
%               not detected.
%
%   diffMessage - String message with nice print of all diffs detected.
%               If no diff is detected, this string is empty.
%
